<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Exports\Exam_Question_Export;
use App\Imports\Exam_Question_Import;
use App\Models\Common;
use App\Models\Exam;
use App\Models\Exam_Leaderboard;
use App\Models\Exam_Question;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use Exception;

class ExamController extends Controller
{
    private $folder = "exam";
    public $common;
    public function __construct()
    {
        $this->common = new Common;
    }

    public function index(Request $request)
    {
        try {

            $params['data'] = [];
            if ($request->ajax()) {

                $query = Exam::query();

                $input_search = $request['input_search'];
                if ($input_search != null) {
                    $query->where('name', 'LIKE', "%{$input_search}%");
                }
                $data = $query->orderby('status', 'desc')->latest()->get();

                $this->common->imageNameToUrl($data, 'image', $this->folder);

                return DataTables()::of($data)
                    ->addIndexColumn()
                    ->addColumn('action', function ($row) {

                        $exam_delete = __('label.delete_exam');

                        $delete = '<form onsubmit="return confirm(\'' . $exam_delete . '\');" method="POST" action="' . route('admin.exam.destroy', [$row->id]) . '">
                            <input type="hidden" name="_token" value="' . csrf_token() . '">
                            <input type="hidden" name="_method" value="DELETE">
                            <button type="submit" class="edit-delete-btn" style="outline: none;"><i class="fa-solid fa-trash-can fa-xl"></i></button></form>';

                        $btn = '<div class="d-flex justify-content-around">';
                        $btn .= '<a href="' . route('admin.exam.question.index', [$row->id]) . '" class="edit-delete-btn mr-2">';
                        $btn .= '<i class="fa-solid fa-circle-question fa-xl"></i>';
                        $btn .= '</a>';
                        $btn .= '<a href="' . route('admin.exam.leaderboard', [$row->id]) . '" class="edit-delete-btn mr-2">';
                        $btn .= '<i class="fa-solid fa-gauge fa-xl"></i>';
                        $btn .= '</a>';
                        $btn .= '<a class="edit-delete-btn mr-2 edit_exam" data-toggle="modal" href="#EditModel" data-id="' . $row->id . '" data-name="' . $row->name . '" data-image="' . $row->image . '" data-total_questions="' . $row->total_questions . '" data-quiz_duration="' . $row->quiz_duration . '" data-entry_fee="' . $row->entry_fee . '" data-min_winning_percentage="' . $row->min_winning_percentage . '" data-winning_coin="' . $row->winning_coin . '">';
                        $btn .= '<i class="fa-solid fa-pen-to-square fa-xl"></i>';
                        $btn .= '</a>';
                        $btn .= $delete;
                        $btn .= '</a></div>';
                        return $btn;
                    })
                    ->addColumn('status', function ($row) {
                        if ($row->status == 1) {
                            $showLabel = __('label.show');
                            return "<button type='button' id='$row->id' onclick='change_status($row->id)' class='show-btn'>$showLabel</button>";
                        } else {
                            $hideLabel = __('label.hide');
                            return "<button type='button' id='$row->id' onclick='change_status($row->id)' class='hide-btn'>$hideLabel</button>";
                        }
                    })
                    ->rawColumns(['action', 'status'])
                    ->make(true);
            }
            return view('admin.exam.index', $params);
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|min:2',
                'entry_fee' => 'required|numeric|min:0',
                'total_questions' => 'required|numeric|min:1',
                'quiz_duration' => 'required|numeric|min:1',
                'image' => 'image|mimes:jpeg,png,jpg|max:5120',
            ]);
            if ($validator->fails()) {
                $errs = $validator->errors()->all();
                return response()->json(['status' => 400, 'errors' => $errs]);
            }

            $requestData = $request->all();
            if (isset($requestData['image'])) {
                $file = $requestData['image'];
                $requestData['image'] = $this->common->saveImage($file, $this->folder, 'exam_');
            } else {
                $requestData['image'] = '';
            }
            $requestData['min_winning_percentage'] = $request['min_winning_percentage'] ?? 0;
            $requestData['winning_coin'] = $request['winning_coin'] ?? 0;
            $requestData['status'] = 1;

            $data = Exam::updateOrCreate(['id' => $requestData['id']], $requestData);
            if (isset($data['id'])) {
                return response()->json(['status' => 200, 'success' => __('label.success_add_exam')]);
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.error_add_exam')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function update($id, Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|min:2',
                'entry_fee' => 'required|numeric|min:0',
                'total_questions' => 'required|numeric|min:1',
                'quiz_duration' => 'required|numeric|min:1',
                'image' => 'image|mimes:jpeg,png,jpg|max:5120',
            ]);
            if ($validator->fails()) {
                $errs = $validator->errors()->all();
                return response()->json(['status' => 400, 'errors' => $errs]);
            }

            $requestData = $request->all();
            if (isset($requestData['image'])) {
                $file = $requestData['image'];
                $requestData['image'] = $this->common->saveImage($file, $this->folder, 'exam_');

                $this->common->deleteImageToFolder($this->folder, basename($requestData['old_image']));
            }
            unset($requestData['old_image']);
            $requestData['min_winning_percentage'] = $request['min_winning_percentage'] ?? 0;
            $requestData['winning_coin'] = $request['winning_coin'] ?? 0;

            $data = Exam::updateOrCreate(['id' => $requestData['id']], $requestData);
            if (isset($data['id'])) {
                return response()->json(['status' => 200, 'success' => __('label.success_edit_exam')]);
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.error_edit_exam')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function destroy($id)
    {
        try {

            $data = Exam::where('id', $id)->first();
            if ($data) {

                // Question
                $question = Exam_Question::where('exam_id', $id)->get();
                if (count($question) > 0) {

                    for ($i = 0; $i < count($question); $i++) {
                        $this->common->deleteImageToFolder($this->folder, $question[$i]['image']);
                        $question[$i]->delete();
                    }
                }

                $this->common->deleteImageToFolder($this->folder, $data['image']);
                $data->delete();
            }
            return redirect()->route('admin.exam.index')->with('success', __('label.exam_delete'));
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function show($id)
    {
        try {

            $data = Exam::where('id', $id)->first();
            if ($data) {

                $data['status'] = $data['status'] === 1 ? 0 : 1;
                $data->save();
                return response()->json(['status' => 200, 'success' => __('label.status_changed'), 'status_code' => $data['status']]);
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.data_not_found')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function question_index($id, Request $request)
    {
        try {

            $params['exam_id'] = $id;
            if ($request->ajax()) {

                $input_search = $request['input_search'];

                $query = Exam_Question::where('exam_id', $id);
                if ($input_search) {
                    $query->where(function ($q) use ($input_search) {
                        $q->where('question', 'LIKE', "%{$input_search}%")
                            ->orWhere('option_a', 'LIKE', "%{$input_search}%")
                            ->orWhere('option_b', 'LIKE', "%{$input_search}%")
                            ->orWhere('option_c', 'LIKE', "%{$input_search}%")
                            ->orWhere('option_d', 'LIKE', "%{$input_search}%");
                    });
                }
                $data = $query->latest()->get();

                $this->common->imageNameToUrl($data, 'image', $this->folder);

                return DataTables()::of($data)
                    ->addIndexColumn()
                    ->addColumn('action', function ($row) {

                        $question_delete = __('label.delete_question');

                        $delete = '<form onsubmit="return confirm(\'' . $question_delete . '\');" method="POST" action="' . route('admin.exam.question.delete', [$row->exam_id, $row->id]) . '">
                        <input type="hidden" name="_token" value="' . csrf_token() . '">
                        <button type="submit" class="edit-delete-btn" style="outline: none;"><i class="fa-solid fa-trash-can fa-xl"></i></button></form>';

                        $btn = '<div class="d-flex justify-content-around">';
                        $btn .= '<a class="edit-delete-btn mr-2 edit_exam_question" data-toggle="modal" href="#EditModel" data-id="' . $row->id . '" data-question="' . $row->question . '" data-image="' . $row->image . '" data-option_a="' . $row->option_a . '" data-option_b="' . $row->option_b . '" data-option_c="' . $row->option_c . '" data-option_d="' . $row->option_d . '" data-correct_answer="' . $row->correct_answer . '" data-note="' . $row->note . '">';
                        $btn .= '<i class="fa-solid fa-pen-to-square fa-xl"></i>';
                        $btn .= '</a>';
                        $btn .= $delete;
                        $btn .= '</div>';
                        return $btn;
                    })
                    ->addColumn('status', function ($row) {
                        if ($row->status == 1) {
                            $showLabel = __('label.show');
                            return "<button type='button' id='$row->id' onclick='change_status($row->id)' class='show-btn'>$showLabel</button>";
                        } else {
                            $hideLabel = __('label.hide');
                            return "<button type='button' id='$row->id' onclick='change_status($row->id)' class='hide-btn'>$hideLabel</button>";
                        }
                    })
                    ->rawColumns(['action', 'status'])
                    ->make(true);
            }
            return view('admin.exam.question', $params);
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function question_save(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'exam_id' => 'required',
                'question' => 'required',
                'option_a' => 'required',
                'option_b' => 'required',
                'option_c' => 'required',
                'option_d' => 'required',
                'correct_answer' => 'required',
                'image' => 'image|mimes:jpeg,png,jpg|max:5120',
            ]);
            if ($validator->fails()) {
                $errs = $validator->errors()->all();
                return response()->json(['status' => 400, 'errors' => $errs]);
            }

            $question = new Exam_Question();
            $question['exam_id'] = $request['exam_id'];
            $question['image'] = '';
            if (isset($request['image'])) {
                $file = $request['image'];
                $question['image'] = $this->common->saveImage($file, $this->folder, 'exam_question_');
            }
            $question['question'] = $request['question'];
            $question['option_a'] = $request['option_a'];
            $question['option_b'] = $request['option_b'];
            $question['option_c'] = $request['option_c'];
            $question['option_d'] = $request['option_d'];
            $question['correct_answer'] = $request['correct_answer'];
            $question['note'] = $request['note'] ?? '';
            $question['status'] = 1;
            if ($question->save()) {
                return response()->json(['status' => 200, 'success' => __('label.success_add_question')]);
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.error_add_question')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function question_update(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'question' => 'required',
                'option_a' => 'required',
                'option_b' => 'required',
                'option_c' => 'required',
                'option_d' => 'required',
                'correct_answer' => 'required',
                'image' => 'image|mimes:jpeg,png,jpg|max:5120',
            ]);
            if ($validator->fails()) {
                $errs = $validator->errors()->all();
                return response()->json(['status' => 400, 'errors' => $errs]);
            }

            $question = Exam_Question::where('id', $request['id'])->first();
            if (isset($question)) {

                if (isset($request['image'])) {
                    $file = $request['image'];
                    $question['image'] = $this->common->saveImage($file, $this->folder, 'exam_question_');

                    $this->common->deleteImageToFolder($this->folder, basename($request['old_image']));
                }
                $question['question'] = $request['question'];
                $question['option_a'] = $request['option_a'];
                $question['option_b'] = $request['option_b'];
                $question['option_c'] = $request['option_c'];
                $question['option_d'] = $request['option_d'];
                $question['correct_answer'] = $request['correct_answer'];
                $question['note'] = $request['note'] ?? '';
                if ($question->save()) {
                    return response()->json(['status' => 200, 'success' => __('label.success_edit_question')]);
                } else {
                    return response()->json(['status' => 400, 'errors' => __('label.error_edit_question')]);
                }
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.error_edit_question')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function question_delete($exam_id, $id)
    {
        try {

            $question = Exam_Question::where('id', $id)->first();
            if (isset($question)) {
                $this->common->deleteImageToFolder($this->folder, $question['image']);
                $question->delete();
            }
            return redirect()->route('admin.exam.question.index', $exam_id)->with('success', __('label.question_delete'));
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function question_show($id)
    {
        try {

            $data = Exam_Question::where('id', $id)->first();
            if (isset($data)) {

                $data['status'] = $data['status'] === 1 ? 0 : 1;
                $data->save();
                return response()->json(['status' => 200, 'success' => __('label.status_changed'), 'status_code' => $data->status]);
            } else {
                return response()->json(['status' => 400, 'errors' => __('label.data_not_found')]);
            }
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    // Import-Export
    public function export(Request $request)
    {
        try {

            return Excel::download(new Exam_Question_Export(), 'Exam-Question-Formate.csv');
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    public function import(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'exam_id' => 'required|numeric',
                'import_file' => 'required|file|mimetypes:text/plain,text/csv,application/csv',
            ]);
            if ($validator->fails()) {
                $errs = $validator->errors()->all();
                return response()->json(['status' => 400, 'errors' => $errs]);
            }

            Excel::import(new Exam_Question_Import($request['exam_id']), $request->file('import_file'));

            return response()->json(['status' => 200, 'success' => __('label.success_import_file')]);
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
    // Leaderboard
    public function leaderboard($id, Request $request)
    {
        try {
            $params['exam_id'] = $id;
            $params['user'] = User::latest()->get();

            if ($request->ajax()) {

                $input_user = $request['input_user'];

                $query = Exam_Leaderboard::where('exam_id', $id);
                if ($input_user != 0) {
                    $query->where('user_id', $input_user);
                }
                $data = $query->with('user')->latest()->get();

                return DataTables()::of($data)
                    ->addIndexColumn()
                    ->addColumn('status', function ($row) {
                        if ($row->status == 1) {
                            $showLabel = __('label.passed');
                            return "<button type='button' class='show-btn'>$showLabel</button>";
                        } else {
                            $hideLabel = __('label.failed');
                            return "<button type='button' class='hide-btn'>$hideLabel</button>";
                        }
                    })
                    ->rawColumns(['status'])
                    ->make(true);
            }
            return view('admin.exam.leaderboard', $params);
        } catch (Exception $e) {
            return response()->json(['status' => 400, 'errors' => $e->getMessage()]);
        }
    }
}
